<?php

namespace App\Http\Controllers;

use App\Models\Ginventarios;
use App\Http\Controllers\Controller;
use App\Http\Requests\StoreGinventariosRequest;
use App\Http\Requests\UpdateGinventariosRequest;
use App\Imports\InventariosImportGeneral;
use App\Models\Gcategorias;
use App\Models\Gsedes;
use App\Models\Gsubcategorias;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Maatwebsite\Excel\Facades\Excel;

class GinventariosController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $inventarios = Ginventarios::all();
        foreach ($inventarios as $inventario) {
            $inventario->subcategorias->categorias->sedes;
        }
        return response()->json($inventarios, 200);
    }

    /** */
    public function getCat(Request $request): JsonResponse
    {
        $perPage = $request->input('per_page', 10); // Número de items por página, default 10
        
        if ($request->categoria === '*') {
            $inventarios = Ginventarios::paginate($perPage);
        } else {
            $inventarios = Ginventarios::where('subcategoria_id', '=', $request->categoria)
                                     ->paginate($perPage);
        }

        // Cargar las relaciones para cada elemento en la colección paginada
        $inventarios->getCollection()->transform(function ($inventario) {
            $inventario->subcategorias->categorias->sedes;
            return $inventario;
        });

        return response()->json($inventarios, 200);
    }

    public function getData(): JsonResponse
    {
        $response = [
            'sedes' => Gsedes::all(),
            'categorias' => Gcategorias::all(),
            'subcategorias' => Gsubcategorias::all(),
        ];
        return response()->json($response, 200);
    }

    public function crearQrs(): JsonResponse
    {
        $inventarios = Ginventarios::whereNull('qr')->get();
        foreach ($inventarios as $inventario) {
            QrCode::format('png')->generate(env('APP_URL') . '/inventario/general?id=' . $inventario->id, '../public/qrcodes/general/' . $inventario->id . '.png');
            $inventario->qr = env('QR_URL') . 'general/' . $inventario->id . '.png';
            $inventario->save();
        }
        return response()->json($inventarios, 200);
    }

    public function importData(Request $request): JsonResponse
    {
        if ($request->hasFile('excel')) {
            $array = Excel::toArray(new InventariosImportGeneral, $request->file('excel'));
            $response = [];
            foreach ($array[0] as $element) {
                $inventario = new Ginventarios();
                $sede = Gsedes::firstWhere('nombre', $element['sede']);
                if ($sede == NULL) {
                    $sede = new Gsedes;
                    $sede->nombre = $element['sede'];
                    $sede->save();
                }
                $categoria = Gcategorias::firstWhere('nombre', $element['categoria']);
                if ($categoria == NULL) {
                    $categoria = new Gcategorias();
                    $categoria->nombre = $element['categoria'];
                    $categoria->sede_id = $sede['id'];
                    $categoria->save();
                }
                $subcategoria = Gsubcategorias::firstWhere('nombre', $element['subcategoria']);
                if ($subcategoria == NULL) {
                    $subcategoria = new Gsubcategorias();
                    $subcategoria->nombre = $element['subcategoria'];
                    $subcategoria->categoria_id = $categoria['id'];
                    $subcategoria->save();
                }
                $inventario->subcategoria_id = $subcategoria->id;
                $inventario->dotacion = $element['dotacion'];
                $inventario->cantidad = $element['cantidad'];
                $inventario->save();
                QrCode::format('png')->generate(env('APP_URL') . '/inventario/general?id=' . $inventario->id, '../public/qrcodes/general/' . $inventario->id . '.png');
                $inventario->qr = env('QR_URL') . 'general/' . $inventario->id . '.png';
                $inventario->save();
                $response[] = $inventario;
            }
            return response()->json($response, 200);
        }
        return response()->json(['message' => "No se encontro el archivo"], 404);
    }

    public function exportBase(Request $request)
    {
        return response()->streamDownload(function () {
    ob_end_clean(); // Elimina cualquier salida previa
    flush(); // Limpia el buffer de salida
    readfile(public_path('cargaArchivosMasivosGeneral.xlsx'));
}, 'cargaArchivosMasivos.xlsx', [
    'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
]);
    }
    public function downloadQr(Request $request)
    {
        return response()->download(public_path('qrcodes/general/'. $request->id . '.png'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreGinventariosRequest $request)
    {
        try {
            $inventario = new Ginventarios();
            $inventario->fill($request->all());
            $inventario->saveOrFail();
            QrCode::format('png')->generate(env('APP_URL') . '/inventario/general?id=' . $inventario->id, '../public/qrcodes/' . $inventario->id . '.png');
            $inventario->qr = env('QR_URL') . 'general/' . $inventario->id . '.png';
            $inventario->saveOrFail();
            return response()->json(['message' => 'Inventario agregado exitosamente', 'product' => $inventario], 201);
        } catch (ModelNotFoundException $exception) {
            return response()->json(['message' => $exception->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Ginventarios $ginventario)
    {
        $ginventario->subcategorias->categorias->sedes;
        return response()->json($ginventario, 200);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Ginventarios $ginventarios)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateGinventariosRequest $request, Ginventarios $ginventario)
    {
        $ginventario->fill($request->all());
        if ($ginventario->save()) {
            return response()->json(['data' => $ginventario, 'message' => 'El inventario se ha actualizado correctamente'], 200);
        }
        return response()->json(['data' => $ginventario, 'message' => 'Ha ocurrido un error al actualizar el inventario'], 500);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Ginventarios $ginventario)
    {
        $ginventario->delete();
        return response()->json(['data' => $ginventario, 'message' => 'El inventario se ha eliminado correctamente'], 200);
    }
}
